/**
 * Background Service Worker
 * 
 * Manages:
 * - Side panel behavior
 * - WebSocket connection to backend
 * - Message routing between content scripts and side panel
 * - Allowlist state
 */

// Extension version - keep in sync with manifest.json and version.js
const EXTENSION_VERSION = '0.1.0';

// Open side panel when extension icon is clicked
chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true });

// State
let ws = null;
let allowlist = new Set();
let isRecording = false;
let currentTabId = null;

// Load allowlist from storage
chrome.storage.local.get(['allowlist'], (result) => {
  if (result.allowlist) {
    allowlist = new Set(result.allowlist);
  }
});

/**
 * Connect to backend WebSocket
 */
function connectWebSocket() {
  if (ws && ws.readyState === WebSocket.OPEN) return;
  
  const wsUrl = 'wss://workflow-observer.agenticintelligence.workers.dev/api/session';
  
  try {
    ws = new WebSocket(wsUrl);
    
    ws.onopen = () => {
      console.log('[background] WebSocket connected');
      broadcastToSidePanel({ type: 'ws_status', connected: true });
    };
    
    ws.onmessage = (event) => {
      const data = JSON.parse(event.data);
      console.log('[background] Received:', data.type);
      
      // Forward to side panel
      broadcastToSidePanel(data);
    };
    
    ws.onclose = () => {
      console.log('[background] WebSocket disconnected');
      broadcastToSidePanel({ type: 'ws_status', connected: false });
      ws = null;
      
      // Reconnect after delay
      setTimeout(connectWebSocket, 5000);
    };
    
    ws.onerror = (error) => {
      console.error('[background] WebSocket error:', error);
    };
  } catch (error) {
    console.error('[background] Failed to connect:', error);
  }
}

/**
 * Send message to backend
 */
function sendToBackend(message) {
  if (ws && ws.readyState === WebSocket.OPEN) {
    ws.send(JSON.stringify(message));
  } else {
    console.warn('[background] WebSocket not connected');
  }
}

/**
 * Broadcast message to side panel
 */
function broadcastToSidePanel(message) {
  chrome.runtime.sendMessage(message).catch(() => {
    // Side panel might not be open
  });
}

/**
 * Check if URL is in allowlist
 */
function isAllowlisted(url) {
  try {
    const hostname = new URL(url).hostname;
    return allowlist.has(hostname);
  } catch {
    return false;
  }
}

/**
 * Handle messages from content scripts and side panel
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'event':
      // Forward ARIA event to backend
      if (isRecording && isAllowlisted(message.data.context.url)) {
        sendToBackend({ type: 'event', data: message.data });
      }
      break;
      
    case 'correction':
      // Forward correction to backend
      sendToBackend({ type: 'correction', prompt: message.prompt });
      break;
      
    case 'edit':
      // Forward edit to backend
      sendToBackend({ type: 'edit', ...message });
      break;
      
    case 'toggle_recording':
      isRecording = message.enabled;
      broadcastToSidePanel({ type: 'recording_status', recording: isRecording });
      break;
      
    case 'add_to_allowlist':
      allowlist.add(message.hostname);
      chrome.storage.local.set({ allowlist: [...allowlist] });
      sendResponse({ success: true });
      break;
      
    case 'remove_from_allowlist':
      allowlist.delete(message.hostname);
      chrome.storage.local.set({ allowlist: [...allowlist] });
      sendResponse({ success: true });
      break;
      
    case 'get_allowlist':
      sendResponse({ allowlist: [...allowlist] });
      break;
      
    case 'get_status':
      sendResponse({
        recording: isRecording,
        connected: ws && ws.readyState === WebSocket.OPEN,
        allowlist: [...allowlist],
        version: EXTENSION_VERSION,
      });
      break;
      
    case 'get_version':
      sendResponse({ version: EXTENSION_VERSION });
      break;
      
    case 'connect':
      connectWebSocket();
      break;
  }
  
  return true; // Keep channel open for async response
});

/**
 * Track active tab for recording indicator
 */
chrome.tabs.onActivated.addListener(async (activeInfo) => {
  currentTabId = activeInfo.tabId;
  
  try {
    const tab = await chrome.tabs.get(activeInfo.tabId);
    if (tab.url && isAllowlisted(tab.url)) {
      // Update badge to show we're watching this site
      chrome.action.setBadgeText({ text: isRecording ? 'REC' : '' });
      chrome.action.setBadgeBackgroundColor({ color: '#ff0000' });
    } else {
      chrome.action.setBadgeText({ text: '' });
    }
  } catch (error) {
    // Tab might have been closed
  }
});

/**
 * Update badge when URL changes
 */
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (tabId === currentTabId && changeInfo.url) {
    if (isAllowlisted(changeInfo.url)) {
      chrome.action.setBadgeText({ text: isRecording ? 'REC' : '' });
      chrome.action.setBadgeBackgroundColor({ color: '#ff0000' });
    } else {
      chrome.action.setBadgeText({ text: '' });
    }
  }
});
