/**
 * Content Script - ARIA Event Observer
 * 
 * Captures semantic user actions via accessibility tree.
 * Sends ARIA-based events (role, name, ref) not raw DOM details.
 */

(function() {
  'use strict';
  
  // Element ref counter
  let refCounter = 0;
  const refs = new Map();
  
  // Expose refs for potential future automation
  window.__workflowRefs = refs;
  
  /**
   * Infer ARIA role from element
   */
  function inferRole(el) {
    // Explicit role takes precedence
    const explicitRole = el.getAttribute('role');
    if (explicitRole) return explicitRole;
    
    const tag = el.tagName.toLowerCase();
    
    // Map common elements to roles
    const roleMap = {
      'button': 'button',
      'a': 'link',
      'input': getInputRole(el),
      'select': 'combobox',
      'textarea': 'textbox',
      'img': 'img',
      'nav': 'navigation',
      'main': 'main',
      'header': 'banner',
      'footer': 'contentinfo',
      'aside': 'complementary',
      'form': 'form',
      'table': 'table',
      'ul': 'list',
      'ol': 'list',
      'li': 'listitem',
    };
    
    return roleMap[tag] || 'generic';
  }
  
  /**
   * Get role for input elements based on type
   */
  function getInputRole(el) {
    const type = el.type?.toLowerCase() || 'text';
    
    const inputRoles = {
      'checkbox': 'checkbox',
      'radio': 'radio',
      'range': 'slider',
      'button': 'button',
      'submit': 'button',
      'reset': 'button',
      'search': 'searchbox',
    };
    
    return inputRoles[type] || 'textbox';
  }
  
  /**
   * Get accessible name for element
   */
  function getAccessibleName(el) {
    // Priority order per ARIA spec
    
    // 1. aria-labelledby
    const labelledBy = el.getAttribute('aria-labelledby');
    if (labelledBy) {
      const label = document.getElementById(labelledBy);
      if (label) return label.textContent?.trim().slice(0, 100);
    }
    
    // 2. aria-label
    const ariaLabel = el.getAttribute('aria-label');
    if (ariaLabel) return ariaLabel.slice(0, 100);
    
    // 3. For inputs, check associated label
    if (el.tagName === 'INPUT' || el.tagName === 'SELECT' || el.tagName === 'TEXTAREA') {
      if (el.id) {
        const label = document.querySelector(`label[for="${el.id}"]`);
        if (label) return label.textContent?.trim().slice(0, 100);
      }
      // Check wrapping label
      const parentLabel = el.closest('label');
      if (parentLabel) {
        const text = parentLabel.textContent?.replace(el.value || '', '').trim();
        if (text) return text.slice(0, 100);
      }
    }
    
    // 4. For images, use alt text
    if (el.tagName === 'IMG') {
      return el.alt?.slice(0, 100) || null;
    }
    
    // 5. For links/buttons, use text content
    if (el.tagName === 'A' || el.tagName === 'BUTTON' || el.getAttribute('role') === 'button') {
      const text = el.textContent?.trim();
      if (text) return text.slice(0, 100);
    }
    
    // 6. Title attribute
    if (el.title) return el.title.slice(0, 100);
    
    // 7. Placeholder for inputs
    if (el.placeholder) return el.placeholder.slice(0, 100);
    
    // 8. Value for buttons
    if (el.tagName === 'INPUT' && (el.type === 'button' || el.type === 'submit')) {
      return el.value?.slice(0, 100) || null;
    }
    
    return null;
  }
  
  /**
   * Find the meaningful interactive element
   */
  function findInteractiveAncestor(el) {
    const interactiveSelector = 
      'button, a, input, select, textarea, ' +
      '[role="button"], [role="link"], [role="menuitem"], [role="tab"], ' +
      '[role="checkbox"], [role="radio"], [role="switch"], [role="option"], ' +
      '[role="treeitem"], [role="listitem"][onclick], [onclick], [tabindex]';
    
    return el.closest(interactiveSelector);
  }
  
  /**
   * Get ARIA info for an element
   */
  function getAriaInfo(el) {
    const interactive = findInteractiveAncestor(el);
    if (!interactive) return null;
    
    const role = inferRole(interactive);
    const name = getAccessibleName(interactive);
    
    // Skip if no meaningful name
    if (!name) return null;
    
    // Create ref for this element
    const ref = `e${++refCounter}`;
    refs.set(ref, interactive);
    
    // Clean up old refs (keep last 100)
    if (refs.size > 100) {
      const keys = [...refs.keys()];
      for (let i = 0; i < keys.length - 100; i++) {
        refs.delete(keys[i]);
      }
    }
    
    return {
      ref,
      role,
      name,
      value: interactive.value || null,
      checked: interactive.checked ?? null,
      expanded: interactive.getAttribute('aria-expanded') ?? null,
      selected: interactive.getAttribute('aria-selected') ?? null,
    };
  }
  
  /**
   * Get nearest landmark for context
   */
  function getNearestLandmark(el) {
    const landmark = el.closest(
      '[role="main"], [role="navigation"], [role="banner"], [role="contentinfo"], ' +
      '[role="complementary"], [role="region"][aria-label], ' +
      'main, nav, header, footer, aside'
    );
    
    if (!landmark) return null;
    
    return landmark.getAttribute('aria-label') || 
           landmark.getAttribute('role') || 
           landmark.tagName.toLowerCase();
  }
  
  /**
   * Build event context
   */
  function getContext() {
    return {
      url: location.href,
      title: document.title,
    };
  }
  
  /**
   * Send event to background script
   */
  function sendEvent(action, element, extra = {}) {
    const aria = getAriaInfo(element);
    if (!aria) return; // Skip non-interactive or unnamed elements
    
    const event = {
      ts: Date.now(),
      action,
      element: aria,
      context: {
        ...getContext(),
        landmark: getNearestLandmark(element),
      },
      ...extra,
    };
    
    chrome.runtime.sendMessage({ type: 'event', data: event }).catch(() => {
      // Extension context might be invalidated
    });
  }
  
  /**
   * Handle click events
   */
  document.addEventListener('click', (e) => {
    // Skip if modifier keys held (likely navigation)
    if (e.ctrlKey || e.metaKey || e.shiftKey) return;
    
    sendEvent('click', e.target);
  }, true);
  
  /**
   * Handle input changes (debounced)
   */
  const inputDebounce = new Map();
  
  document.addEventListener('input', (e) => {
    const el = e.target;
    if (!el.matches('input, textarea, select')) return;
    
    // Clear existing timeout
    const existing = inputDebounce.get(el);
    if (existing) clearTimeout(existing);
    
    // Debounce - only send after 500ms of no typing
    inputDebounce.set(el, setTimeout(() => {
      inputDebounce.delete(el);
      
      // Don't capture password values
      if (el.type === 'password') {
        sendEvent('input', el, { valueMasked: true });
      } else {
        // Capture value for non-sensitive inputs
        sendEvent('input', el);
      }
    }, 500));
  }, true);
  
  /**
   * Handle form submissions
   */
  document.addEventListener('submit', (e) => {
    const form = e.target;
    if (!form.matches('form')) return;
    
    // Get submit button if one was clicked
    const submitBtn = form.querySelector('[type="submit"]:focus, button:focus');
    
    sendEvent('submit', submitBtn || form);
  }, true);
  
  /**
   * Handle navigation (popstate, hashchange)
   */
  window.addEventListener('popstate', () => {
    // Send navigation event
    chrome.runtime.sendMessage({
      type: 'event',
      data: {
        ts: Date.now(),
        action: 'navigate',
        element: { role: 'navigation', name: 'Browser back/forward' },
        context: getContext(),
      }
    }).catch(() => {});
  });
  
  /**
   * Track URL changes via History API
   */
  const originalPushState = history.pushState;
  const originalReplaceState = history.replaceState;
  
  history.pushState = function(...args) {
    originalPushState.apply(this, args);
    
    chrome.runtime.sendMessage({
      type: 'event',
      data: {
        ts: Date.now(),
        action: 'navigate',
        element: { role: 'navigation', name: 'Page navigation' },
        context: getContext(),
      }
    }).catch(() => {});
  };
  
  history.replaceState = function(...args) {
    originalReplaceState.apply(this, args);
    // Don't emit for replaceState - usually not user-initiated
  };
  
  console.log('[workflow-observer] Content script loaded');
})();
