/**
 * Side Panel UI
 * 
 * Renders workflow diagrams and handles user interaction.
 * Pure renderer - all logic is in background.js and backend.
 */

// Initialize Mermaid
mermaid.initialize({
  startOnLoad: false,
  theme: 'neutral',
  securityLevel: 'loose',
  flowchart: {
    useMaxWidth: true,
    htmlLabels: true,
    curve: 'basis',
  },
});

// DOM Elements
const elements = {
  recordingIndicator: document.getElementById('recording-indicator'),
  siteName: document.getElementById('site-name'),
  toggleBtn: document.getElementById('toggle-btn'),
  settingsBtn: document.getElementById('settings-btn'),
  connectionBar: document.getElementById('connection-bar'),
  connectionStatus: document.getElementById('connection-status'),
  workflowInfo: document.getElementById('workflow-info'),
  workflowName: document.getElementById('workflow-name'),
  workflowCount: document.getElementById('workflow-count'),
  diagram: document.getElementById('diagram'),
  correctionInput: document.getElementById('correction-input'),
  sendBtn: document.getElementById('send-btn'),
  settingsModal: document.getElementById('settings-modal'),
  closeSettings: document.getElementById('close-settings'),
  allowlist: document.getElementById('allowlist'),
  newSite: document.getElementById('new-site'),
  addSiteBtn: document.getElementById('add-site-btn'),
  versionNumber: document.getElementById('version-number'),
};

// State
let state = {
  recording: false,
  connected: false,
  currentWorkflow: null,
  currentStep: null,
};

/**
 * Initialize
 */
async function init() {
  // Get initial status
  chrome.runtime.sendMessage({ type: 'get_status' }, (response) => {
    if (response) {
      state.recording = response.recording;
      state.connected = response.connected;
      updateUI();
      renderAllowlist(response.allowlist);
      if (response.version) {
        elements.versionNumber.textContent = response.version;
      }
    }
  });
  
  // Connect to backend
  chrome.runtime.sendMessage({ type: 'connect' });
  
  // Set up event listeners
  setupEventListeners();
}

/**
 * Set up UI event listeners
 */
function setupEventListeners() {
  // Toggle recording
  elements.toggleBtn.addEventListener('click', () => {
    state.recording = !state.recording;
    chrome.runtime.sendMessage({ 
      type: 'toggle_recording', 
      enabled: state.recording 
    });
    updateUI();
  });
  
  // Settings modal
  elements.settingsBtn.addEventListener('click', () => {
    elements.settingsModal.classList.remove('hidden');
    refreshAllowlist();
  });
  
  elements.closeSettings.addEventListener('click', () => {
    elements.settingsModal.classList.add('hidden');
  });
  
  // Add site to allowlist
  elements.addSiteBtn.addEventListener('click', addSite);
  elements.newSite.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') addSite();
  });
  
  // Send correction
  elements.sendBtn.addEventListener('click', sendCorrection);
  elements.correctionInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendCorrection();
    }
  });
  
  // Close modal on backdrop click
  elements.settingsModal.addEventListener('click', (e) => {
    if (e.target === elements.settingsModal) {
      elements.settingsModal.classList.add('hidden');
    }
  });
}

/**
 * Listen for messages from background
 */
chrome.runtime.onMessage.addListener((message) => {
  switch (message.type) {
    case 'ws_status':
      state.connected = message.connected;
      updateConnectionStatus();
      break;
      
    case 'recording_status':
      state.recording = message.recording;
      updateUI();
      break;
      
    case 'render':
      renderDiagram(message.mermaid, message.current_step);
      break;
      
    case 'status':
      updateWorkflowInfo(message.workflow_name, message.seen_count);
      break;
      
    case 'toast':
      showToast(message.message);
      break;
      
    case 'config':
      // Backend sent config on connect
      console.log('[panel] Config received:', message);
      break;
  }
});

/**
 * Update UI based on state
 */
function updateUI() {
  // Recording indicator
  if (state.recording) {
    elements.recordingIndicator.classList.add('recording');
    elements.toggleBtn.textContent = 'Stop';
    elements.toggleBtn.classList.add('recording');
    elements.siteName.textContent = 'Recording...';
  } else {
    elements.recordingIndicator.classList.remove('recording');
    elements.toggleBtn.textContent = 'Start';
    elements.toggleBtn.classList.remove('recording');
    elements.siteName.textContent = 'Not recording';
  }
  
  updateConnectionStatus();
}

/**
 * Update connection status bar
 */
function updateConnectionStatus() {
  if (state.connected) {
    elements.connectionBar.classList.remove('disconnected');
    elements.connectionBar.classList.add('connected');
    elements.connectionStatus.textContent = 'Connected';
  } else {
    elements.connectionBar.classList.remove('connected');
    elements.connectionBar.classList.add('disconnected');
    elements.connectionStatus.textContent = 'Disconnected - Reconnecting...';
  }
}

/**
 * Update workflow info section
 */
function updateWorkflowInfo(name, seenCount) {
  if (name) {
    elements.workflowInfo.classList.remove('hidden');
    elements.workflowName.textContent = name;
    elements.workflowCount.textContent = seenCount ? `seen ${seenCount}x` : '';
  } else {
    elements.workflowInfo.classList.add('hidden');
  }
}

/**
 * Render Mermaid diagram
 */
async function renderDiagram(mermaidCode, currentStep) {
  if (!mermaidCode) {
    elements.diagram.innerHTML = '<p class="placeholder">Start recording to see your workflow</p>';
    return;
  }
  
  try {
    // Generate unique ID for this render
    const id = `diagram-${Date.now()}`;
    
    // Render Mermaid
    const { svg } = await mermaid.render(id, mermaidCode);
    elements.diagram.innerHTML = svg;
    
    // Highlight current step
    if (currentStep) {
      highlightCurrentStep(currentStep);
    }
    
    // Make nodes clickable for editing
    setupNodeClickHandlers();
    
  } catch (error) {
    console.error('[panel] Mermaid render error:', error);
    elements.diagram.innerHTML = `<p class="placeholder">Error rendering diagram</p>`;
  }
}

/**
 * Highlight the current step in the diagram
 */
function highlightCurrentStep(stepId) {
  // Mermaid generates nodes with IDs like "flowchart-step-1-0"
  const nodes = elements.diagram.querySelectorAll('.node');
  
  nodes.forEach((node) => {
    node.classList.remove('current');
    
    // Check if this node matches the current step
    const nodeId = node.id || node.getAttribute('data-id');
    if (nodeId && nodeId.includes(stepId)) {
      node.classList.add('current');
    }
  });
}

/**
 * Set up click handlers on diagram nodes
 */
function setupNodeClickHandlers() {
  const nodes = elements.diagram.querySelectorAll('.node');
  
  nodes.forEach((node) => {
    node.style.cursor = 'pointer';
    
    node.addEventListener('click', () => {
      const nodeId = extractStepId(node);
      if (nodeId) {
        showNodeEditMenu(node, nodeId);
      }
    });
  });
}

/**
 * Extract step ID from Mermaid node
 */
function extractStepId(node) {
  const id = node.id || '';
  // Mermaid generates IDs like "flowchart-A-0" where A is the node label
  const match = id.match(/flowchart-([^-]+)/);
  return match ? match[1] : null;
}

/**
 * Show edit menu for a node (placeholder - implement as needed)
 */
function showNodeEditMenu(node, stepId) {
  // TODO: Implement node editing UI
  // For now, just log
  console.log('[panel] Node clicked:', stepId);
  showToast(`Editing step: ${stepId}`);
}

/**
 * Send correction to backend
 */
function sendCorrection() {
  const prompt = elements.correctionInput.value.trim();
  if (!prompt) return;
  
  chrome.runtime.sendMessage({ type: 'correction', prompt });
  elements.correctionInput.value = '';
  showToast('Correction sent');
}

/**
 * Add site to allowlist
 */
function addSite() {
  let hostname = elements.newSite.value.trim();
  if (!hostname) return;
  
  // Clean up hostname
  hostname = hostname.replace(/^https?:\/\//, '').replace(/\/.*$/, '');
  
  chrome.runtime.sendMessage({ 
    type: 'add_to_allowlist', 
    hostname 
  }, () => {
    elements.newSite.value = '';
    refreshAllowlist();
    showToast(`Added ${hostname}`);
  });
}

/**
 * Remove site from allowlist
 */
function removeSite(hostname) {
  chrome.runtime.sendMessage({ 
    type: 'remove_from_allowlist', 
    hostname 
  }, () => {
    refreshAllowlist();
    showToast(`Removed ${hostname}`);
  });
}

/**
 * Refresh allowlist display
 */
function refreshAllowlist() {
  chrome.runtime.sendMessage({ type: 'get_allowlist' }, (response) => {
    if (response?.allowlist) {
      renderAllowlist(response.allowlist);
    }
  });
}

/**
 * Render allowlist in settings
 */
function renderAllowlist(sites) {
  if (!sites || sites.length === 0) {
    elements.allowlist.innerHTML = '<li class="empty">No sites added yet</li>';
    return;
  }
  
  elements.allowlist.innerHTML = sites.map(site => `
    <li>
      <span>${site}</span>
      <button class="remove-btn" data-site="${site}">×</button>
    </li>
  `).join('');
  
  // Add click handlers
  elements.allowlist.querySelectorAll('.remove-btn').forEach((btn) => {
    btn.addEventListener('click', () => {
      removeSite(btn.dataset.site);
    });
  });
}

/**
 * Show toast notification
 */
function showToast(message) {
  const existing = document.querySelector('.toast');
  if (existing) existing.remove();
  
  const toast = document.createElement('div');
  toast.className = 'toast';
  toast.textContent = message;
  document.body.appendChild(toast);
  
  setTimeout(() => toast.remove(), 3000);
}

// Initialize on load
init();
